package com.boxity.components;

import java.io.*;

import org.apache.tapestry.AbstractComponent;
import org.apache.tapestry.IMarkupWriter;
import org.apache.tapestry.IRequestCycle;
import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.DefaultMethodRetryHandler;
import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * Tapestry component used to include static content from a
 * relative or absolute URI into the HTML response.
 * This component may be used to emulate server-side include functionality.
 * Since relative links of the source URI may not be meaningful within the context
 * of the application, it may be best to avoid them.
 * <p/>
 * Caching is enabled by default but may be disabled to force the component to
 * fetch the target URI contents on each rendering.
 * <p/>
 * Logging level of DEBUG will cause retrieved data to be logged.
 *
 * @author Shawn M Church
 * @version Include, Jun 21, 2005, 2:02:33 PM
 */

public abstract class Include extends AbstractComponent
{
   private final static Log _log = LogFactory.getLog(Include.class);

   public abstract Object getUri();

   public abstract String getStyleClass();

   public abstract boolean getCache();

   /**
    * Resolves and prints the data stream specified by the uri parameter.
    *
    * @param writer
    * @param cycle
    */
   protected void renderComponent(IMarkupWriter writer, IRequestCycle cycle)
   {
      if( cycle.isRewinding() )
         return;

      Object uri = getUri();

      if( uri == null )
         return;

      String key = "boxity_include_" + uri;
      String responseBody = null;
      boolean shouldStore = false;

      // Retrieve the cached data if specified.
      if( getCache() )
      {
         try
         {
            responseBody = ( String ) cycle.getEngine().getPool().retrieve(key);
         }

            // Fine, ignore the exception.
         catch ( Exception e )
         {
         }
      }

      if( responseBody == null )
      {
         if( getCache() )
            shouldStore = true;

         try
         {
            HttpClient client = new HttpClient();

            // Create a GET method instance.
            GetMethod method = new GetMethod(uri.toString());

            // Provide custom retry handler as necessary
            DefaultMethodRetryHandler retryhandler = new DefaultMethodRetryHandler();
            retryhandler.setRequestSentRetryEnabled(false);
            retryhandler.setRetryCount(3);
            method.setMethodRetryHandler(retryhandler);

            try
            {
               // Execute the method.
               int statusCode = client.executeMethod(method);

               if( statusCode != HttpStatus.SC_OK )
               {
                  _log.error("Method failed: " + method.getStatusLine());
               }

               // Deal with the response.
               // todo: ensure correct character encoding and is not binary data
               responseBody = method.getResponseBodyAsString();
               _log.debug(responseBody);
            }
            catch ( IOException e )
            {
               _log.error("Failed to get response.");
               e.printStackTrace();
            }
            finally
            {
               // Release the connection.
               method.releaseConnection();
            }
         }

         catch ( Exception e )
         {
            responseBody = null;
            _log.error(e);
         }
      }

      if( responseBody == null )
         return;

      // Cache the fetched data stream if specified.
      if( getCache() && shouldStore )
      {
         try
         {
            cycle.getEngine().getPool().store(key, responseBody);
         }

            // Fine, ignore
         catch ( Exception e )
         {
         }
      }

      String styleClass = getStyleClass();

      if( styleClass != null )
      {
         writer.begin("span");
         writer.attribute("class", styleClass);

         renderInformalParameters(writer, cycle);
      }

      writer.printRaw(responseBody);

      if( styleClass != null )
         writer.end(); // <span>
   }
}
